import asyncio
import importlib
import json
import logging

from asyncio import Task

from datetime import datetime

from typing import TextIO

from pylog.pylogger import PyLogger

from py_pli.pylib import URPCInterfaceType
from py_pli.pylib import URPCFunctions
from py_pli.pylib import VUnits
from py_pli.pylib import send_msg

from gccore import gccore_main

from virtualunits.vu_node_application import VUNodeApplication

from urpc_enum.error import FirmwareErrorsDic

from urpc.barcodereaderfunctions import BarcodeReaderFunctions
from urpc.corexymoverfunctions import CoreXYMoverFunctions
from urpc.dualmoverfunctions import DualMoverFunctions
from urpc.fancontrolfunctions import FanControlFunctions
from urpc.measurementfunctions import MeasurementFunctions
from urpc.moverfunctions import MoverFunctions
from urpc.nodefunctions import NodeFunctions
from urpc.serialfunctions import SerialFunctions
from urpc.stackerliftmoverfunctions import StackerLiftMoverFunctions
from urpc.systemcontrolfunctions import SystemControlFunctions
from urpc.temperaturecontrolfunctions import TemperatureControlFunctions

from fleming.common.node_io import *


### Endpoint Utility Functions #########################################################################################

# Node Endpoint ########################################################################################################

node_endpoints = {
    'eef'       : {'id':0x0008, 'delay':5, 'mot_file':'EEF_Node.mot'},
    'mc6'       : {'id':0x0010, 'delay':1, 'mot_file':'MC6_Node.mot'},
    'fmb'       : {'id':0x0020, 'delay':1, 'mot_file':'FMB_Node.mot'},
    'mc6_stk'   : {'id':0x0028, 'delay':1, 'mot_file':'MC6_Node.mot'},
    'pmc'       : {'id':0x00F0, 'delay':1, 'mot_file':'PMC_Node.mot'},
    'pmc1'      : {'id':0x00F1, 'delay':1, 'mot_file':'PMC_Node.mot'},
    'pmc2'      : {'id':0x00F2, 'delay':1, 'mot_file':'PMC_Node.mot'},
    'pmc3'      : {'id':0x00F3, 'delay':1, 'mot_file':'PMC_Node.mot'},
}

def get_node_endpoint(name) -> NodeFunctions:
    """ Getter for node endpoints. """
    can_id = node_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'NodeFunctions')
    return endpoint


def get_node_unit(endpoint) -> VUNodeApplication:
    """ Getter for node virtual unit. """
    can_id = node_endpoints[endpoint]['id']
    for node in VUnits.instance.hal.nodes.values():
        if node.canID == can_id:
            return node
        
    raise Exception(f"No virtual unit configured for node endpoint '{endpoint}' with CAN-ID: 0x{can_id:04X}")


async def start_firmware(node_name):
    """ Start the firmware of a node. Does nothing if the firmware is already running. """
    node = get_node_endpoint(node_name)
    if (await node.GetFirmwareInfo())[0] != 1:
        await node.StartFirmware()
        await asyncio.sleep(node_endpoints[node_name]['delay'])
        if (await node.GetFirmwareInfo())[0] != 1:
            raise Exception(f"Failed to start the firmware.")
        status, error = await node.GetStatus()
        if status != 2:
            raise Exception(f"Firmware failed to initialize with error = 0x{error:04X} : {FirmwareErrorsDic.get(error, 'Unknown Error')}")
    
    return f"start_firmware() done"


async def update_firmware(node_name, mot_file=None):
    """ Update the firmware of a node. """
    if mot_file is None:
        mot_file = node_endpoints[node_name]['mot_file']

    PyLogger.logger.info(f"Updating {node_name.upper()} Firmware '{mot_file}'")
    node_unit = get_node_unit(node_name)
    firmware_updater = VUnits.instance.hal.firmware_updater
    await firmware_updater.flash_node(node_unit, mot_file, node_unit.Name, device=1, force=True)

    return f"update_firmware() done"


async def update_bootloader(node_name, mot_file='Bootloader.mot'):
    """ Update the bootloader of a node. """
    PyLogger.logger.info(f"Updating {node_name.upper()} Bootloader '{mot_file}'")
    node_unit = get_node_unit(node_name)
    firmware_updater = VUnits.instance.hal.firmware_updater
    await firmware_updater.flash_node(node_unit, mot_file, node_unit.Name, device=0, force=True)

    return f"update_bootloader() done"


async def update_fpga(mot_file='EEF_FPGA.mot'):
    """ Update the FPGA firmware of the EEF board. """
    PyLogger.logger.info(f"Updating FPGA '{mot_file}'")
    node_unit =  get_node_unit('eef')
    firmware_updater = VUnits.instance.hal.firmware_updater
    await firmware_updater.flash_node(node_unit, mot_file, node_unit.Name, device=2, force=True)

    return f"update_fpga() done"


# Mover Endpoint ########################################################################################################

mover_endpoints = {
    'as1'       : {'id':0x010A},  # EEF M3 Aperture Slider 1 (FPGA)
    'as2'       : {'id':0x010B},  # EEF M4 Aperture Slider 2 (FPGA)
    'fms'       : {'id':0x0110},  # MC6 M1 Filder Module Slider (fast)
    'usfm'      : {'id':0x0111},  # MC6 M2 US Lum Focus Mover (fast)
    'fm'        : {'id':0x0112},  # MC6 M3 Focus Mover (slow)
    'bld'       : {'id':0x0113},  # MC6 M4 Bottom Light Director (slow)
    'pd'        : {'id':0x0114},  # MC6 M5 Plate Door (slow)
    'els'       : {'id':0x0115},  # MC6 M6 Excitation Light Selector (slow)
    'lrm'       : {'id':0x012A},  # Stacker Left Release Mover (slow)
    'rrm'       : {'id':0x012B},  # Stacker Right Release Mover (slow)
    'zdrive'    : {'id':0x01F0},  # Z-Drive of PMC (fast)
    'pipettor'  : {'id':0x02F0},  # Pipettor of PMC (fast)
    # Alternative Naming for MC6 Tests
    'mc6m1'     : {'id':0x0110},  # MC6 M1 (fast)
    'mc6m2'     : {'id':0x0111},  # MC6 M2 (fast)
    'mc6m3'     : {'id':0x0112},  # MC6 M3 (slow)
    'mc6m4'     : {'id':0x0113},  # MC6 M4 (slow)
    'mc6m5'     : {'id':0x0114},  # MC6 M5 (slow)
    'mc6m6'     : {'id':0x0115},  # MC6 M6 (slow)
}

def get_mover_endpoint(name) -> MoverFunctions:
    """ Getter for mover endpoints. """
    can_id = mover_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'MoverFunctions')
    return endpoint


# CoreXY Endpoint ######################################################################################################

corexy_mover_endpoints = {
    'st'    : {'id':0x0108},   # Scan Table
}

def get_corexy_mover_endpoint(name='st') -> CoreXYMoverFunctions:
    """ Getter for corexy endpoints. The name is optional, since we only have one endpoint. """
    can_id = corexy_mover_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'CoreXYMoverFunctions')
    return endpoint


# Dual Endpoint ########################################################################################################

dual_mover_endpoints = {
    'st'    : {'id':0x0108},   # Scan Table
}

def get_dual_mover_endpoint(name='st') -> DualMoverFunctions:
    """ Getter for dual mover endpoints. """
    can_id = dual_mover_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'DualMoverFunctions')
    return endpoint


# Stacker Lift Mover Endpoint ##########################################################################################

stacker_lift_mover_endpoints = {
    'llm'   : {'id':0x0128},  # Stacker Left Lift Mover 
    'rlm'   : {'id':0x0129},  # Stacker Right Lift Mover 
}

def get_stacker_lift_mover_endpoint(name) -> StackerLiftMoverFunctions:
    """ Getter for stacker lift mover endpoints. """
    can_id = stacker_lift_mover_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'StackerLiftMoverFunctions')
    return endpoint


# Measurement Endpoint #################################################################################################

measurement_endpoints = {
    'meas'  : {'id':0x010C},
} 

def get_measurement_endpoint(name='meas') -> MeasurementFunctions:
    """ Getter for measurement endpoints. The name is optional, since we only have one endpoint. """
    can_id = measurement_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'MeasurementFunctions')
    return endpoint


async def fpga_mem_dump():
    """ Save the contents of the FPGAs memory. """
    meas = get_measurement_endpoint()
    size = 58
    data = []
    for address in range(0x0000, 0x5000, size):
        if (address + size) > 0x5000:
            size = 0x5000 - address
        data.extend(await meas.Read(address, size))

    with open(f"fpga_mem_{datetime.now().strftime('%Y%m%d_%H%M%S')}.mem", 'w') as file:
        file.write(f"@0000\r\n")
        for address in range(len(data)):
            file.write(f"{data[address]:04X}")
            if (address & 0x000F) < 0x000F:
                file.write(' ')
            else:
                file.write('\r\n')

    return f"fpga_mem_dump() done"


async def fpga_sfr_dump():
    """ Save the contents of the FPGAs special function registers. """
    meas = get_measurement_endpoint()

    with open(f"fpga_sfr_{datetime.now().strftime('%Y%m%d_%H%M%S')}.txt", 'w') as file:
        file.write(f"sfr  ; data\n")
        for address in range(256):
            data = (await meas.ReadSpecialFunctionRegister(address))[0]
            file.write(f"0x{address:02X} ; 0x{data:02X}\n")

    return f"fpga_sfr_dump() done"


# Serial Endpoint ######################################################################################################

serial_endpoints = {
    'trf'   : {'id':0x010E, 'channel':0},
}

def get_serial_endpoint(name) -> SerialFunctions:
    """ Getter for serial endpoints. """
    can_id = serial_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'SerialFunctions')
    return endpoint


# Barcode Reader Endpoint ##############################################################################################

barcode_reader_endpoints = {
    'bcr1'  : {'id':0x010D, 'channel':0},
    'bcr2'  : {'id':0x010F, 'channel':0},
    'bcr3'  : {'id':0x010F, 'channel':1},
    'bcr4'  : {'id':0x010F, 'channel':2},
    'bcr5'  : {'id':0x010F, 'channel':3},
}

def get_barcode_reader_endpoint(name) -> BarcodeReaderFunctions:
    """ Getter for barcode reader endpoints. """
    can_id = barcode_reader_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'BarcodeReaderFunctions')
    return endpoint


# System Control Endpoint ##############################################################################################

system_control_endpoints = {
    'sys'   : {'id':0x0120},
}

def get_system_control_endpoint(name='sys') -> SystemControlFunctions:
    """ Getter for system control endpoints. The name is optional, since we only have one endpoint. """
    can_id = system_control_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'SystemControlFunctions')
    return endpoint


# Fan Control Endpoint #################################################################################################

fan_control_endpoints = {
    'eef_fan'   : {'id':0x020A},
    'fmb_fan'   : {'id':0x0121},
}

def get_fan_control_endpoint(name='fmb_fan') -> FanControlFunctions:
    """ Getter for fan control endpoints. """
    can_id = fan_control_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'FanControlFunctions')
    return endpoint


# Temperature Control Endpoint #########################################################################################

temperature_control_endpoints = {
    'eef_tc'    : {'id':0x0209},
    'fmb_tc'    : {'id':0x0122},
}

def get_temperature_control_endpoint(name) -> TemperatureControlFunctions:
    """ Getter for temperature control endpoints. """
    can_id = temperature_control_endpoints[name]['id']
    endpoint = create_endpoint(can_id, 'TemperatureControlFunctions')
    return endpoint


# Endpoint Creator #####################################################################################################

def create_endpoint(can_id, endpoint_class_name):
    """ Create endpoints that are not supported by the PyRunner yet. """
    endpoint_creator = URPCFunctions.instance.endPointCreator

    endpoint_module = importlib.import_module('urpc.' + endpoint_class_name.lower())
    endpoint_class = getattr(endpoint_module, endpoint_class_name)

    endpoint = None
    
    if can_id in endpoint_creator.urpcFunctions.endPointsDic:
        PyLogger.logger.debug(f"Endpoint already exist for CAN ID: {can_id}")
        endpoint = endpoint_creator.urpcFunctions.endPointsDic[can_id]

    if not isinstance(endpoint, endpoint_class):
        PyLogger.logger.debug(f"New Endpoint created for CAN ID: {can_id}")
        endpoint = endpoint_class(can_id, endpoint_creator.transmitterQueueParent, endpoint_creator.receptionQueueChild, endpoint_creator.iso15765xSend, endpoint_creator.event_loop)

        endpoint_creator.creationEvent.clear()
        endpoint_creator.createEndpointQueueParent.send(can_id)
        endpoint_creator.creationEvent.wait()
        endpoint_creator.creationEvent.clear()

        endpoint_creator.urpcFunctions.endPointsDic[can_id] = endpoint

    return endpoint


### Logging ############################################################################################################

TEST_INFO = 28  # WARNING = 30 > TEST_INFO = 28 > INFO = 20

logging.addLevelName(TEST_INFO, 'TEST_INFO')

def log_test_info(msg, *args, **kwargs):
    """ Write log entry with the custom TEST_INFO logging level. """
    PyLogger.logger.logger.log(TEST_INFO, msg, *args, **kwargs)


def set_logging_level(level):
    """ Change the PyRunner logging level during runtime. """
    PyLogger.logger.logger.setLevel(level)


### Ground Control #####################################################################################################

def is_ground_control():
    """ Check wheather the current task is executed by ground control. """
    return isinstance(gccore_main.running_task, Task) and (gccore_main.running_task.done() is False)


async def send_gc_msg(msg: str, log: bool = False, report: TextIO = None):
    """ Send a message to ground control. Optionally write the message into the log or a report file. """
    if log:
        PyLogger.logger.info(msg)
    if report:
        report.write(msg + '\n')

    await send_msg(json.dumps({'result': msg}))


async def write_gc_msg(file: TextIO, msg: str):
    """ Send a message to ground control and write it into a text file. """
    file.write(msg + '\n')
    await send_gc_msg(msg)


async def send_graph(header, results, maximum, rows=8):
    """ Send a ascii art graph to ground control when called by ground control or write it into the log otherwise. """
    ground_control = is_ground_control()
    if ground_control:
        background = '\u25FB'
        line = '\u25FC'
    else:
        background = '\033[90m' + '-' + '\033[0m'
        line = '0'

    values = [round(result / maximum * (rows-1)) for result in results]
        
    chart = [[background]*len(values) for i in range(rows)]
    for x, value in enumerate(values):
        for y in range((rows-1), ((rows-1) - min(value, (rows-1)) - 1), -1):
            chart[y][x] = line

    graph = ''
    for y in range(rows):
        graph += f"{round(maximum * (1 - y / (rows-1))):02d} {''.join(chart[y])}\n"
    
    if ground_control:
        await send_gc_msg(header + '\n' + graph)
    else:
        PyLogger.logger.info('\n' + header + '\n' + graph)


########################################################################################################################
